<?php

namespace App\Http\Controllers;

use App\Models\Pegawai;
use App\Models\Presensi;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;

class ImportPresensiController extends Controller
{
    public function form()
    {
        return view('presensi.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:csv,xlsx,xls',
        ]);

        $data = Excel::toArray([], $request->file('file'));
        $errors = [];

        foreach ($data[0] as $index => $row) {
            if ($index === 0) continue; // Skip header

            $nip     = trim($row[0] ?? '');
            $tanggal = trim($row[1] ?? '');
            $jam     = trim($row[2] ?? '');

            if (!$nip || !$tanggal || !$jam) {
                $errors[] = "Baris ke-" . ($index + 1) . ": Data tidak lengkap.";
                continue;
            }

            $pegawai = Pegawai::where('nip', $nip)->first();
            if (!$pegawai) {
                $errors[] = "Baris ke-" . ($index + 1) . ": NIP {$nip} tidak ditemukan.";
                continue;
            }

            // Konversi jam dan status
            try {
                $jamMasuk = \Carbon\Carbon::createFromFormat('H:i', $jam);
                $jamBatas = \Carbon\Carbon::createFromTime(8, 0);
                $status = $jamMasuk->lessThanOrEqualTo($jamBatas) ? 'Hadir' : 'Telat';
            } catch (\Exception $e) {
                $errors[] = "Baris ke-" . ($index + 1) . ": Format jam tidak valid.";
                continue;
            }

            Presensi::updateOrCreate(
                ['pegawai_id' => $pegawai->id, 'tanggal' => $tanggal],
                ['status' => $status]
            );
        }

        if (count($errors)) {
            return redirect()->back()->withErrors($errors);
        }

        return redirect()->back()->with('success', 'Data fingerprint berhasil diimport.');
    }

    public function hitungHadir(Request $request)
    {
        $request->validate([
            'pegawai_id' => 'required|exists:pegawais,id',
            'bulan' => 'required|date_format:Y-m',
        ]);

        $jumlahHadir = Presensi::where('pegawai_id', $request->pegawai_id)
            ->where('status', 'Hadir')
            ->whereRaw("DATE_FORMAT(tanggal, '%Y-%m') = ?", [$request->bulan])
            ->count();

        return response()->json([
            'jumlah_hadir' => $jumlahHadir,
        ]);
    }
}
