<?php

namespace App\Imports;

use App\Models\Pegawai;
use App\Models\Jabatan;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class PegawaiImport implements ToCollection, WithHeadingRow
{
    protected $berhasil = 0;
    protected $gagal = [];

    public function collection(Collection $rows)
    {
        $jabatanValid = Jabatan::pluck('nama', 'id')->mapWithKeys(function ($nama, $id) {
            return [strtolower(trim($nama)) => $id];
        })->toArray();

        foreach ($rows as $index => $row) {
            $baris = $index + 2; // +2 karena header ada di baris 1

            if (
                empty($row['nama']) || empty($row['nip']) || empty($row['jabatan']) ||
                empty($row['status']) || empty($row['tanggal_masuk']) ||
                empty($row['pendidikan']) || empty($row['divisi']) || empty($row['status_ptkp'])
            ) {
                $this->gagal[] = ["baris" => $baris, "pesan" => "Data tidak lengkap."];
                continue;
            }

            if (Pegawai::where('nip', $row['nip'])->exists()) {
                $this->gagal[] = ["baris" => $baris, "pesan" => "NIP {$row['nip']} sudah terdaftar."];
                Log::warning("Import Pegawai - Duplikat NIP", ['nip' => $row['nip'], 'baris' => $baris]);
                continue;
            }

            $namaJabatan = strtolower(trim($row['jabatan']));
            if (!array_key_exists($namaJabatan, $jabatanValid)) {
                $this->gagal[] = ["baris" => $baris, "pesan" => "Jabatan '{$row['jabatan']}' belum terdaftar."];
                Log::warning("Import Pegawai - Jabatan tidak ditemukan", ['jabatan' => $row['jabatan'], 'baris' => $baris]);
                continue;
            }

            try {
                Pegawai::create([
                    'nama' => $row['nama'],
                    'nip' => $row['nip'],
                    'jabatan_id' => $jabatanValid[$namaJabatan],
                    'status_karyawan' => $row['status'],
                    'tanggal_masuk' => $row['tanggal_masuk'],
                    'pendidikan_terakhir' => $row['pendidikan'],
                    'divisi' => $row['divisi'],
                    'status_ptkp' => $row['status_ptkp'],
                ]);
                $this->berhasil++;
            } catch (\Exception $e) {
                $this->gagal[] = ["baris" => $baris, "pesan" => "Gagal menyimpan data: {$e->getMessage()}"];
                Log::error("Import Pegawai - Gagal menyimpan data", ['baris' => $baris, 'error' => $e->getMessage()]);
            }
        }

        if (!empty($this->gagal)) {
            Session::flash('import_errors', $this->gagal);
        }
    }

    public function getBerhasil()
    {
        return $this->berhasil;
    }

    public function getGagal()
    {
        return $this->gagal;
    }
}